<?php
/**
 * Plugin Name: Orbidi Acees
 * Plugin URI:
 * Description: Soluciona errores de accesibilidad relacionados con los landmarks ARIA en enlaces "skip to content" y garantiza que todo el contenido esté dentro de landmarks.
 * Version: 1.1
 * Author: Orbidi ManuelDev
 * Author URI:
 * License: GPL v2 or later
 */

// Evitar acceso directo al archivo
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Clase principal del plugin
 */
class Fix_Skip_Link_Accessibility {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Iniciar el plugin cuando todos los plugins estén cargados
        add_action('wp_loaded', array($this, 'init'));
    }
    
    /**
     * Inicializa el plugin
     */
    public function init() {
        // Método 1: Envolver el enlace de salto en un landmark
        add_action('wp_footer', array($this, 'wrap_skip_link_in_landmark'), 99);
        
        // Método 2: Eliminar el enlace problemático si existe
        add_action('wp_head', array($this, 'remove_skip_link'), 100);
        
        // Método 3: Agregar un nuevo enlace de salto correctamente estructurado
        add_action('wp_head', array($this, 'add_proper_skip_link'));
        
        // Método 4: Corregir el enlace existente mediante CSS
        add_action('wp_head', array($this, 'fix_skip_link_styles'));
        
        // Método 5: Corregir el enlace .skip-link del tema y asegurar que todo el contenido esté en landmarks
        add_action('wp_footer', array($this, 'fix_standard_theme_skip_link'), 99);
        
        // Método 6: Asegurar que todo el contenido esté dentro de landmarks
        add_action('wp_footer', array($this, 'ensure_content_in_landmarks'), 999);
        
        // Método 7: Garantizar que haya exactamente un punto de referencia main
        add_action('wp_footer', array($this, 'ensure_single_main_landmark'), 990);
        
        // Método 8: Verificar que los iframes tienen a lo sumo un main
        add_action('wp_footer', array($this, 'check_iframes_main_landmark'), 995);
    }
    
    /**
     * Método 1: Envuelve el enlace de salto en un landmark
     */
    public function wrap_skip_link_in_landmark() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const skipLink = document.querySelector('.ea11y-skip-to-content-link');
                if (skipLink && !skipLink.closest('[role="navigation"]')) {
                    // Crear un nuevo elemento nav
                    const navElement = document.createElement('nav');
                    navElement.setAttribute('aria-label', 'Saltar enlaces');
                    
                    // Obtener el padre del enlace
                    const parent = skipLink.parentNode;
                    
                    // Insertar el nav antes del enlace
                    parent.insertBefore(navElement, skipLink);
                    
                    // Mover el enlace dentro del nav
                    navElement.appendChild(skipLink);
                }
            });
        </script>
        <?php
    }
    
    /**
     * Método 2: Eliminar el enlace problemático
     */
    public function remove_skip_link() {
        ?>
        <style>
            .ea11y-skip-to-content-link {
                display: none !important;
            }
        </style>
        <?php
    }
    
    /**
     * Método 3: Añadir un nuevo enlace de salto correctamente estructurado
     */
    public function add_proper_skip_link() {
        ?>
        <style>
            .proper-skip-to-content {
                background-color: #fff;
                color: #0073aa;
                font-weight: 700;
                left: -999em;
                padding: 10px;
                position: absolute;
                text-decoration: none;
                z-index: 999;
            }
            
            .proper-skip-to-content:focus {
                left: 0;
                outline: none;
                position: fixed;
                top: 0;
            }
        </style>
        
        <nav aria-label="Enlaces de accesibilidad">
            <a href="#content" class="proper-skip-to-content">
                Ir al contenido principal
            </a>
        </nav>
        <?php
    }
    
    /**
     * Método 4: Corregir el enlace existente mediante CSS y JS
     */
    public function fix_skip_link_styles() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const skipLink = document.querySelector('.ea11y-skip-to-content-link');
                if (skipLink) {
                    // Eliminar atributos problemáticos
                    skipLink.removeAttribute('role');
                    
                    // Asegurarse de que el enlace esté en un elemento nav
                    if (!skipLink.closest('nav')) {
                        const nav = document.createElement('nav');
                        nav.setAttribute('aria-label', 'Accesibilidad');
                        
                        const parent = skipLink.parentNode;
                        parent.insertBefore(nav, skipLink);
                        nav.appendChild(skipLink);
                    }
                    
                    // Asegurar que el enlace lleve al contenido correcto
                    skipLink.addEventListener('click', function(e) {
                        const contentElem = document.getElementById('content');
                        if (contentElem) {
                            e.preventDefault();
                            contentElem.tabIndex = -1;
                            contentElem.focus();
                            contentElem.scrollIntoView({behavior: 'smooth'});
                        }
                    });
                }
            });
        </script>
        <?php
    }
    
    /**
     * Método 5: Corregir el enlace .skip-link estándar de temas WordPress
     * Soluciona el error específico del enlace con clase .skip-link
     */
    public function fix_standard_theme_skip_link() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                // Buscar el enlace .skip-link que está causando el problema
                const standardSkipLink = document.querySelector('.skip-link');
                
                if (standardSkipLink) {
                    // Eliminar el atributo role="link" que es incorrecto
                    standardSkipLink.removeAttribute('role');
                    
                    // Verificar si el enlace ya está dentro de un landmark (nav)
                    if (!standardSkipLink.closest('nav')) {
                        // Crear un elemento nav que servirá como landmark
                        const navElement = document.createElement('nav');
                        navElement.setAttribute('aria-label', 'Saltar a la navegación');
                        
                        // Conseguir el elemento padre del enlace
                        const parent = standardSkipLink.parentNode;
                        
                        // Insertar el nav antes del enlace
                        parent.insertBefore(navElement, standardSkipLink);
                        
                        // Mover el enlace dentro del nav
                        navElement.appendChild(standardSkipLink);
                    }
                }
            });
        </script>
        <?php
    }
    
    /**
     * Método 6: Asegurar que todo el contenido esté dentro de landmarks
     * Esta es una medida de último recurso que busca contenido huérfano
     * (no incluido en landmarks) y lo envuelve en un elemento con un landmark.
     */
    public function ensure_content_in_landmarks() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                // Lista de selectores de landmarks ARIA
                const landmarkSelectors = [
                    'main', 'nav', 'header', 'footer', 'aside', 'section[aria-label]', 
                    'section[aria-labelledby]', 'article', 'form[aria-label]', 
                    'form[aria-labelledby]', '[role="main"]', '[role="navigation"]', 
                    '[role="banner"]', '[role="contentinfo"]', '[role="complementary"]', 
                    '[role="search"]', '[role="form"]', '[role="region"][aria-label]', 
                    '[role="region"][aria-labelledby]'
                ];
                
                // Construir un selector para todos los elementos con landmark
                const combinedSelector = landmarkSelectors.join(', ');
                
                // Obtener todos los elementos con landmark
                const landmarks = document.querySelectorAll(combinedSelector);
                
                // Obtener el body para verificar sus hijos directos
                const body = document.body;
                
                // Crear una lista de elementos que no están dentro de landmarks
                // pero deberían estarlo (excluyendo scripts, estilos, etc.)
                const nonLandmarkContent = [];
                
                // Verificar elementos directamente en el body
                Array.from(body.children).forEach(function(element) {
                    // Ignorar elementos que ya son landmarks o no deberían estar en un landmark
                    if (
                        element.matches(combinedSelector) || 
                        element.tagName === 'SCRIPT' || 
                        element.tagName === 'STYLE' || 
                        element.tagName === 'LINK' || 
                        element.tagName === 'META' ||
                        element.tagName === 'NOSCRIPT' ||
                        element.getAttribute('aria-hidden') === 'true' ||
                        element.style.display === 'none' ||
                        element.classList.contains('skip-link') ||
                        element.classList.contains('ea11y-skip-to-content-link') ||
                        element.closest(combinedSelector) !== null
                    ) {
                        return;
                    }
                    
                    // Elemento huérfano encontrado, añadirlo a la lista
                    nonLandmarkContent.push(element);
                });
                
                // Si hay contenido huérfano, agrupar los elementos adyacentes
                if (nonLandmarkContent.length > 0) {
                    // Crear grupos de elementos consecutivos huérfanos
                    const groups = [];
                    let currentGroup = [nonLandmarkContent[0]];
                    
                    for (let i = 1; i < nonLandmarkContent.length; i++) {
                        const current = nonLandmarkContent[i];
                        const prev = nonLandmarkContent[i - 1];
                        
                        // Si este elemento sigue directamente al anterior en el DOM
                        if (current.previousElementSibling === prev) {
                            // Añadir al grupo actual
                            currentGroup.push(current);
                        } else {
                            // Finalizar el grupo anterior y empezar uno nuevo
                            groups.push(currentGroup);
                            currentGroup = [current];
                        }
                    }
                    
                    // Añadir el último grupo
                    groups.push(currentGroup);
                    
                    // Procesar cada grupo
                    groups.forEach(function(group, index) {
                        // Crear una sección como landmark
                        const section = document.createElement('section');
                        section.setAttribute('role', 'region');
                        section.setAttribute('aria-label', 'Contenido adicional ' + (index + 1));
                        
                        // Insertar la sección antes del primer elemento del grupo
                        const firstElement = group[0];
                        firstElement.parentNode.insertBefore(section, firstElement);
                        
                        // Mover todos los elementos del grupo dentro de la sección
                        group.forEach(function(element) {
                            section.appendChild(element);
                        });
                    });
                }
            });
        </script>
        <?php
    }
    /**
     * Método 7: Garantizar que el documento tenga exactamente un punto de referencia main
     * - Si no hay ninguno, crea uno
     * - Si hay más de uno, corrige la estructura manteniendo solo uno
     */
    public function ensure_single_main_landmark() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                // Buscar todos los elementos con role="main" o elementos <main>
                const mainLandmarks = document.querySelectorAll('main, [role="main"]');
                
                // Caso 1: No hay ningún landmark main
                if (mainLandmarks.length === 0) {
                    // Buscar el contenedor más probable para ser el contenido principal
                    let contentArea = document.getElementById('content');
                    
                    // Si no existe #content, buscar otras alternativas comunes
                    if (!contentArea) {
                        contentArea = document.getElementById('primary');
                    }
                    if (!contentArea) {
                        contentArea = document.querySelector('.content-area');
                    }
                    if (!contentArea) {
                        contentArea = document.querySelector('.site-content');
                    }
                    
                    if (contentArea) {
                        // Si ya es un elemento HTML semántico que no debe estar dentro de main, 
                        // le añadimos el role="main"
                        if (['HEADER', 'FOOTER', 'NAV', 'ASIDE'].includes(contentArea.tagName)) {
                            contentArea.setAttribute('role', 'main');
                        } else {
                            // Si no es un elemento que no deba estar dentro de main, lo envolvemos
                            const mainElement = document.createElement('main');
                            contentArea.parentNode.insertBefore(mainElement, contentArea);
                            mainElement.appendChild(contentArea);
                        }
                    } else {
                        // Como último recurso, si no encontramos ningún contenedor adecuado,
                        // envolvemos todo el contenido del body (excepto header, footer, etc.) en un main
                        const body = document.body;
                        const mainElement = document.createElement('main');
                        
                        // Lista de elementos que normalmente no van dentro de main
                        const excludedElements = ['HEADER', 'FOOTER', 'NAV', 'ASIDE', 'SCRIPT', 'NOSCRIPT', 'STYLE'];
                        
                        // Mover todos los elementos apropiados al nuevo main
                        let elementsToMove = [];
                        Array.from(body.children).forEach(function(element) {
                            if (!excludedElements.includes(element.tagName) && 
                                !element.matches('.skip-link') && 
                                !element.matches('[role="dialog"]') &&
                                !element.matches('[role="banner"]') &&
                                !element.matches('[role="contentinfo"]')) {
                                elementsToMove.push(element);
                            }
                        });
                        
                        // Insertar el main antes del primer elemento a mover
                        if (elementsToMove.length > 0) {
                            body.insertBefore(mainElement, elementsToMove[0]);
                            // Mover los elementos seleccionados dentro del main
                            elementsToMove.forEach(function(element) {
                                mainElement.appendChild(element);
                            });
                        } else {
                            // Si no hay elementos para mover, simplemente añadir el main al final
                            body.appendChild(mainElement);
                            mainElement.innerHTML = '<p>Contenido principal</p>';
                        }
                    }
                }
                // Caso 2: Hay más de un landmark main
                else if (mainLandmarks.length > 1) {
                    // Mantener solo el primer landmark main y corregir los demás
                    const keepMain = mainLandmarks[0];
                    
                    // Para cada landmark main adicional
                    for (let i = 1; i < mainLandmarks.length; i++) {
                        const extraMain = mainLandmarks[i];
                        
                        // Si es un elemento <main>, cambiar a <div role="region">
                        if (extraMain.tagName === 'MAIN') {
                            // Crear un reemplazo
                            const replacement = document.createElement('div');
                            replacement.setAttribute('role', 'region');
                            replacement.setAttribute('aria-label', 'Contenido adicional');
                            
                            // Copiar atributos excepto role
                            Array.from(extraMain.attributes).forEach(function(attr) {
                                if (attr.name !== 'role') {
                                    replacement.setAttribute(attr.name, attr.value);
                                }
                            });
                            
                            // Mover el contenido
                            while (extraMain.firstChild) {
                                replacement.appendChild(extraMain.firstChild);
                            }
                            
                            // Reemplazar el elemento main
                            extraMain.parentNode.replaceChild(replacement, extraMain);
                        }
                        // Si es otro elemento con role="main", quitar ese role
                        else {
                            extraMain.removeAttribute('role');
                            extraMain.setAttribute('role', 'region');
                            extraMain.setAttribute('aria-label', 'Contenido adicional');
                        }
                    }
                }
            });
        </script>
        <?php
    }
    
    /**
     * Método 8: Verificar que los iframes tienen a lo sumo un punto de referencia main
     * Este método comprueba los iframes accesibles y corrige los que puedan tener
     * múltiples landmarks main.
     */
    public function check_iframes_main_landmark() {
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                // Intentar acceder a todos los iframes en la página
                const iframes = document.querySelectorAll('iframe');
                
                iframes.forEach(function(iframe) {
                    try {
                        // Intentar acceder al contenido del iframe (puede fallar por política de mismo origen)
                        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                        
                        // Si podemos acceder, verificar los landmark main
                        const iframeMainLandmarks = iframeDoc.querySelectorAll('main, [role="main"]');
                        
                        // Si hay más de un main, corregir
                        if (iframeMainLandmarks.length > 1) {
                            // Mantener solo el primer landmark main
                            const keepMain = iframeMainLandmarks[0];
                            
                            // Corregir los demás
                            for (let i = 1; i < iframeMainLandmarks.length; i++) {
                                const extraMain = iframeMainLandmarks[i];
                                
                                // Si es un elemento <main>, cambiar a <div role="region">
                                if (extraMain.tagName === 'MAIN') {
                                    // Crear un reemplazo
                                    const replacement = iframeDoc.createElement('div');
                                    replacement.setAttribute('role', 'region');
                                    replacement.setAttribute('aria-label', 'Contenido adicional del iframe');
                                    
                                    // Copiar atributos excepto role
                                    Array.from(extraMain.attributes).forEach(function(attr) {
                                        if (attr.name !== 'role') {
                                            replacement.setAttribute(attr.name, attr.value);
                                        }
                                    });
                                    
                                    // Mover el contenido
                                    while (extraMain.firstChild) {
                                        replacement.appendChild(extraMain.firstChild);
                                    }
                                    
                                    // Reemplazar el elemento main
                                    extraMain.parentNode.replaceChild(replacement, extraMain);
                                }
                                // Si es otro elemento con role="main", quitar ese role
                                else {
                                    extraMain.removeAttribute('role');
                                    extraMain.setAttribute('role', 'region');
                                    extraMain.setAttribute('aria-label', 'Contenido adicional del iframe');
                                }
                            }
                        }
                    } catch (e) {
                        // Ignorar errores de seguridad por política de mismo origen
                        // Esto ocurre cuando intentamos acceder a iframes de otros dominios
                        console.log('No se puede acceder al iframe para verificar landmarks');
                    }
                });
            });
        </script>
        <?php
    }
}

// Inicializar plugin
new Fix_Skip_Link_Accessibility();